<?php
/**
 * Settings: Emails > Payment Notification
 *
 * @package SimplePay
 * @copyright Copyright (c) 2023, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 4.7.3
 */

namespace SimplePay\Pro\Settings\Emails\PaymentNotification;

use SimplePay\Core\Emails;
use SimplePay\Core\Payments\Payment_Confirmation;
use Simplepay\Core\Settings;
use SimplePay\Pro\Payments\Payment_Confirmation as Pro_Payment_Confirmation;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Registers "Email > Payment Notification" subsection.
 *
 * @since 4.7.3
 *
 * @param \SimplePay\Core\Settings\Subsections_Collection $subsections Subsections collection.
 * @return void
 */
function register_subsection( $subsections ) {
	$email = new Emails\Email\PaymentNotificationEmail();

	$subsections->add(
		new Settings\Subsection(
			array(
				'id'       => $email->get_id(),
				'section'  => 'emails',
				'label'    => esc_html( $email->get_label() ),
				'priority' => 30,
			)
		)
	);
}
add_action(
	'simpay_register_settings_subsections',
	__NAMESPACE__ . '\\register_subsection'
);

/**
 * Registers "Emails > Payment Confirmation" settings.
 *
 * @since 4.7.3
 *
 * @param \SimplePay\Core\Settings\Settings_Collection $settings Settings collection.
 * @return void
 */
function register_settings( $settings ) {
	$email = new Emails\Email\PaymentNotificationEmail();
	$id    = $email->get_id();

	// Enable/disable.
	$settings->add(
		new Settings\Setting_Checkbox(
			array(
				'id'          => sprintf( 'email_%s', $id ),
				'section'     => 'emails',
				'subsection'  => $id,
				'label'       => $email->get_label(),
				'input_label' => $email->get_description(),
				'value'       => $email->is_enabled() ? 'yes' : 'no',
				'priority'    => 10,
				'schema'      => array(
					'type'    => 'string',
					'enum'    => array( 'yes', 'no' ),
					'default' => 'yes',
				),
				'toggles'     => array(
					'value'    => 'yes',
					'settings' => array(
						sprintf( 'email_%s_to', $id ),
						sprintf( 'email_%s_subject', $id ),
						sprintf( 'email_%s_body', $id ),
						sprintf( 'email_%s_test', $id ),
					),
				),
			)
		)
	);

	// To.
	$to = simpay_get_setting(
		sprintf( 'email_%s_to', $id ),
		esc_html( get_bloginfo( 'admin_email' ) )
	);

	$settings->add(
		new Settings\Setting_Input(
			array(
				'id'         => sprintf( 'email_%s_to', $id ),
				'section'    => 'emails',
				'subsection' => $id,
				'label'      => esc_html_x(
					'Send To',
					'setting label',
					'simple-pay'
				),
				'value'      => $to,
				'classes'    => array(
					'regular-text',
				),
				'priority'   => 20,
				'schema'     => array(
					'type'    => 'string',
					'default' => get_option( 'admin_email', '' ),
				),
			)
		)
	);

	// Subject.
	$subject = simpay_get_setting(
		sprintf( 'email_%s_subject', $id ),
		esc_html(
			sprintf(
				/* translators: %s Site name */
				__( 'New Payment on %s', 'simple-pay' ),
				get_bloginfo( 'name' )
			)
		)
	);

	$settings->add(
		new Settings\Setting_Input(
			array(
				'id'         => sprintf( 'email_%s_subject', $id ),
				'section'    => 'emails',
				'subsection' => $id,
				'label'      => esc_html_x(
					'Subject',
					'setting label',
					'simple-pay'
				),
				'value'      => $subject,
				'classes'    => array(
					'regular-text',
				),
				'priority'   => 30,
				'schema'     => array(
					'type' => 'string',
				),
			)
		)
	);

	$settings->add(
		new Settings\Setting(
			array(
				'id'         => sprintf( 'email_%s_body', $id ),
				'section'    => 'emails',
				'subsection' => $id,
				'label'      => esc_html_x(
					'Message',
					'setting label',
					'simple-pay'
				),
				'output'     => function () use ( $id ) {
					wp_editor(
						get_body_setting_or_default(),
						sprintf( 'email_%s_body', $id ),
						array(
							'textarea_name' => sprintf(
								'simpay_settings[email_%s_body]',
								$id
							),
							'textarea_rows' => 10,
						)
					);

					Payment_Confirmation\Template_Tags\__unstable_print_tag_list(
						'',
						array_merge(
							Payment_Confirmation\Template_Tags\__unstable_get_tags_and_descriptions(),
							Pro_Payment_Confirmation\Template_Tags\__unstable_get_tags_and_descriptions()
						)
					);
				},
				'priority'   => 40,
				'schema'     => array(
					'type' => 'string',
				),
			)
		)
	);
}
add_action(
	'simpay_register_settings',
	__NAMESPACE__ . '\\register_settings'
);

/**
 * Returns the email body setting or the default.
 *
 * Installations < 4.7.3 do not have the email settings created upon
 * installation so they need fallback values. Installations >= 4.7.3 are activated
 * with default values saved to the database, so these defaults aren't used.
 *
 * @since 4.7.3
 *
 * @return string Email message body.
 */
function get_body_setting_or_default() {
	$email = new Emails\Email\PaymentNotificationEmail();

	/** @var string $body */
	$body = simpay_get_setting(
		sprintf( 'email_%s_body', $email->get_id() ),
		'A new payment from {customer:email} for {total-amount} has been received for &ldquo;{form-title}&rdquo;.'
	);

	return $body;
}
