<?php
/**
 * Stripe Checkout: Subscription
 *
 * @package SimplePay\Pro\Payments\Stripe_Checkout\Subscription
 * @copyright Copyright (c) 2022, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.6.0
 */

namespace SimplePay\Pro\Payments\Stripe_Checkout\Subscription;

use SimplePay\Pro\Payments\Subscription;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Adds `subscription_data` to Stripe Checkout Session.
 *
 * @since 3.6.0
 *
 * @param array                         $session_args Arguments used to create a PaymentIntent.
 * @param SimplePay\Core\Abstracts\Form $form Form instance.
 * @param array                         $form_data Form data generated by the client.
 * @param array                         $form_values Values of named fields in the payment form.
 * @param int                           $customer_id Stripe Customer ID.
 * @return array
 */
function add_subscription(
	$session_args,
	$form,
	$form_data,
	$form_values,
	$customer_id
) {
	if ( ! Subscription\payment_form_request_has_subscription(
		$form,
		$form_data,
		$form_values
	) ) {
		return $session_args;
	}

	// Retrieve price option.
	$price = simpay_payment_form_prices_get_price_by_id(
		$form,
		$form_data['price']['id']
	);

	if ( false === $price ) {
		throw new \Exception(
			__( 'Unable to locate payment form price.', 'simple-pay' )
		);
	}

	// Remove/reset arguments for Subscriptions.
	$session_args['mode'] = 'subscription';
	unset( $session_args['payment_intent_data'] );

	if ( isset( $session_args['submit_type'] ) ) {
		unset( $session_args['submit_type'] );
	}

	// Build (on-site) Subscription arguments based on the request.
	$subscription_args = Subscription\get_args_from_payment_form_request(
		$form,
		$form_data,
		$form_values,
		$customer_id
	);

	// Merge subscription line items and setup fee line items.
	$session_args['line_items'] = array_merge(
		$subscription_args['items'],
		Subscription\get_setup_fee_from_payment_form_request(
			$form,
			$form_data,
			$form_values,
			$customer_id
		)
	);

	// Set a description to base line item.
	if ( null !== $price->label ) {
		$session_args['line_items'][0]['description'] = $price->label;
	}

	// Add trial length if needed.
	if ( isset( $subscription_args['trial_period_days'] ) ) {
		$session_args['subscription_data']['trial_period_days'] =
			$subscription_args['trial_period_days'];

		unset( $subscription_args['trial_period_days'] );
	}

	// Copy top level metadata to subscription_data.
	$metadata = isset( $session_args['metadata'] )
		? $session_args['metadata']
		: array();

	$session_args['subscription_data']['metadata'] = array_merge(
		$metadata,
		$subscription_args['metadata']
	);

	// If an application fee was added to the top level via the
	// `simpay_get_subscription_args_from_payment_form_request` filter, move
	// it to the `subscription_data` key.
	if ( isset( $session_args['application_fee_percent'] ) ) {
		$session_args['subscription_data']['application_fee_percent'] = $session_args['application_fee_percent'];
		unset( $session_args['application_fee_percent'] );
	}

	return $session_args;
}
// Only update if UPE is not enabled. Otherwise it is handled in the updated `wpsp/__internal__payment` endpoint.
if ( ! simpay_is_upe() ) {
	add_filter(
		'simpay_get_session_args_from_payment_form_request',
		__NAMESPACE__ . '\\add_subscription',
		10,
		5
	);
}
