<?php
/**
 * Stripe Customer
 *
 * @package SimplePay\Core\Payments\Customer
 * @copyright Copyright (c) 2022, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.5.0
 */

namespace SimplePay\Core\Payments\Customer;

use SimplePay\Core\i18n;
use SimplePay\Core\Legacy;
use SimplePay\Core\API\Customers;
use SimplePay\Core\Utils;

use Exception;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Retrieves a Customer.
 *
 * @since 3.7.0
 *
 * @param string|array $customer Customer ID or {
 *   Arguments used to retrieve a Customer.
 *
 *   @type string $id Customer ID.
 * }
 * @param array        $api_request_args {
 *   Additional request arguments to send to the Stripe API when making a request.
 *
 *   @type string $api_key API Secret Key to use.
 * }
 * @return \SimplePay\Vendor\Stripe\Customer
 */
function retrieve( $customer, $api_request_args = array() ) {
	_deprecated_function(
		__FUNCTION__,
		'4.1.0',
		'\SimplePay\Core\API\Customers\retrieve'
	);

	return Customers\retrieve( $customer, $api_request_args );
}

/**
 * Creates a Customer record.
 *
 * @since 3.6.0
 *
 * @param array $customer_args Optional arguments used to create a customer.
 * @param array $api_request_args {
 *   Additional request arguments to send to the Stripe API when making a request.
 *
 *   @type string $api_key API Secret Key to use.
 * }
 * @return \SimplePay\Vendor\Stripe\Customer
 */
function create( $customer_args = array(), $api_request_args = array() ) {
	_deprecated_function(
		__FUNCTION__,
		'4.1.0',
		'\SimplePay\Core\API\Customers\create'
	);

	return Customers\create( $customer_args, $api_request_args );
}

/**
 * Updates a Customer record.
 *
 * @since 3.7.0
 *
 * @param string $customer_id ID of the Customer to update.
 * @param array  $customer_args Data to update Customer with.
 * @param array  $api_request_args {
 *   Additional request arguments to send to the Stripe API when making a request.
 *
 *   @type string $api_key API Secret Key to use.
 * }
 * @return \SimplePay\Vendor\Stripe\Customer $customer Stripe Customer.
 */
function update( $customer_id, $customer_args, $api_request_args = array() ) {
	_deprecated_function(
		__FUNCTION__,
		'4.1.0',
		'\SimplePay\Core\API\Customers\update'
	);

	return Customers\update( $customer_id, $customer_args, $api_request_args );
}

/**
 * Deletes a Customer record.
 *
 * @since 3.8.0
 *
 * @param string $customer_id ID of the Customer to update.
 * @param array  $api_request_args {
 *   Additional request arguments to send to the Stripe API when making a request.
 *
 *   @type string $api_key API Secret Key to use.
 * }
 * @return \SimplePay\Vendor\Stripe\Customer $customer Stripe Customer.
 */
function delete( $customer_id, $api_request_args = array() ) {
	_deprecated_function(
		__FUNCTION__,
		'4.1.0',
		'\SimplePay\Core\API\Customers\delete'
	);

	return Customers\delete( $customer_id, $api_request_args );
}

/**
 * Retrieves Customers.
 *
 * @since 3.9.0
 *
 * @param array $args Optional arguments used when listing Customers.
 * @param array $api_request_args {
 *   Additional request arguments to send to the Stripe API when making a request.
 *
 *   @type string $api_key API Secret Key to use.
 * }
 * @return \SimplePay\Vendor\Stripe\Customers[]
 */
function all( $args = array(), $api_request_args = array() ) {
	_deprecated_function(
		__FUNCTION__,
		'4.1.0',
		'\SimplePay\Core\API\Customers\all'
	);

	return Customers\all( $args, $api_request_args );
}

/**
 * Uses the current payment form request to generate arguments for a Customer.
 *
 * @since 3.6.0
 *
 * @param SimplePay\Core\Abstracts\Form $form Form instance.
 * @param array                         $form_data Form data generated by the client.
 * @param array                         $form_values Values of named fields in the payment form.
 * @return array
 */
function get_args_from_payment_form_request( $form, $form_data, $form_values ) {
	$customer_args = array();

	$defaults = array(
		'name'     => null,
		'phone'    => null,
		'email'    => null,
		'metadata' => null,
		'tax'      => array(
			'ip_address' => Utils\get_current_ip_address(),
		),
	);

	$customer_args = wp_parse_args( $customer_args, $defaults );

	// Attach coupon to metadata.
	if ( isset( $form_values['simpay_coupon'] ) && ! empty( $form_values['simpay_coupon'] ) ) {
		$customer_args['coupon'] = sanitize_text_field( $form_values['simpay_coupon'] );

		// Clear Stripe object cache so dynamic values are available.
		// @todo implement cache clearing within Stripe_Object_Query_Trait
		// when it is available in this namespace.
		delete_transient( 'simpay_stripe_' . $customer_args['coupon'] );
	}

	// Attach email.
	if ( isset( $form_values['simpay_email'] ) ) {
		if ( is_array( $form_values['simpay_email'] ) ) {
			$form_values['simpay_email'] = end( $form_values['simpay_email'] );
		}

		$customer_args['email'] = sanitize_text_field( $form_values['simpay_email'] );
	}

	// Attach name.
	if ( isset( $form_values['simpay_customer_name'] ) ) {
		if ( is_array( $form_values['simpay_customer_name'] ) ) {
			$form_values['simpay_customer_name'] = end( $form_values['simpay_customer_name'] );
		}

		$customer_args['name'] = sanitize_text_field( $form_values['simpay_customer_name'] );
	}

	// Attach phone number.
	if ( isset( $form_values['simpay_telephone'] ) ) {
		$customer_args['phone'] = sanitize_text_field( $form_values['simpay_telephone'] );
	}

	// Attach a Tax ID.
	if ( isset( $form_values['simpay_tax_id'] ) ) {
		$tax_id_type = isset( $form_values['simpay_tax_id_type'] )
			? $form_values['simpay_tax_id_type']
			: '';

		$valid_tax_id_types = i18n\get_stripe_tax_id_types();

		if ( false === array_key_exists( $tax_id_type, $valid_tax_id_types ) ) {
			throw new Exception(
				esc_html__( 'Please select a valid Tax ID type.', 'simple-pay' )
			);
		}

		$tax_id = sanitize_text_field( $form_values['simpay_tax_id'] );

		$customer_args['tax_id_data'] = array(
			array(
				'type'  => $tax_id_type,
				'value' => $tax_id,
			),
		);
	}

	// Attach billing address.
	$billing_address = simpay_get_form_address_data( 'billing', $form_values );

	if ( $billing_address && ! empty( $billing_address ) ) {
		$customer_args['address'] = $billing_address;
	}

	// Remove null values, Stripe doesn't like them.
	// Do this before Shipping, because we need a value for Shipping Name.
	$customer_args = array_filter(
		$customer_args,
		function ( $var ) {
			return ! is_null( $var );
		}
	);

	// Attach shipping address.
	$field            = isset( $form_values['simpay_same_billing_shipping'] ) ? 'billing' : 'shipping';
	$shipping_address = simpay_get_form_address_data( $field, $form_values );

	if ( $shipping_address && ! empty( $shipping_address ) ) {
		$customer_args['shipping']['address'] = $shipping_address;
		$customer_args['shipping']['name']    = isset( $customer_args['name'] ) ? $customer_args['name'] : '';
		$customer_args['shipping']['phone']   = isset( $customer_args['phone'] ) ? $customer_args['phone'] : '';
	}

	/**
	 * Filters arguments used to create a Customer from a payment form request.
	 *
	 * @since 3.6.0
	 *
	 * @param array                         $customer_args
	 * @param SimplePay\Core\Abstracts\Form $form Form instance.
	 * @param array                         $form_data Form data generated by the client.
	 * @param array                         $form_values Values of named fields in the payment form.
	 * @return array
	 */
	$customer_args = apply_filters(
		'simpay_get_customer_args_from_payment_form_request',
		$customer_args,
		$form,
		$form_data,
		$form_values
	);

	return $customer_args;
}
