<?php
/*
  Name: AW Plugin License Manager
  Version: 2.2
  Author: Aakash Chakravarthy
  Author URI: https://www.aakashweb.com/
*/

namespace SC_PRO;

defined( 'ABSPATH' ) || exit;

class AW_Plugin_License_Manager{

    /** @const The activation API server endpoint */
    const API_SERVER = 'https://www.aakashweb.com/wp-json/aw-api/v1/';

    /** @var This AW plugin license manager version */
    private $license_manager_version = '2.2';

    /** @var The ID of the plugin, SKU */
    private $plugin_id;

    /** @var The name of the plugin */
    private $plugin_name;

    /** @var The version of the plugin */
    private $plugin_version;

    /** @var The plugin file information. Used for update information */
    private $plugin_file;

    /** @var The plugin prefix to use. Used for saving DB options */
    private $plugin_prefix;

    /** @var The plugin slug . Used for update information */
    private $plugin_slug;

    /** @var The domain URL where the plugin is running  */
    private $plugin_domain;

    /** @var The admin page where the license form is displayed. Used by admin notice */
    private $license_page;

    /** @var The plugin homepage where the license can be bought. Used by admin notice */
    private $home_url;

    /** @var The mode of operation switch for production/development */
    private $dev;
    
    /**
     * Constructor, initializes the class variables and sets the license manager hooks
     */
    public function __construct( $prop ){
        
        $this->plugin_id = $prop[ 'id' ];
        $this->plugin_name = $prop[ 'name' ];
        $this->plugin_version = $prop[ 'version' ];
        $this->plugin_file = $prop[ 'file' ];
        $this->plugin_prefix = $prop[ 'prefix' ];
        $this->plugin_slug = $prop[ 'slug' ];
        $this->plugin_domain = $this->domain_url();
        $this->license_page = $prop[ 'license_page' ];
        $this->home_url = $prop[ 'home_url' ];
        $this->dev = isset( $prop[ 'dev' ] ) && $prop[ 'dev' ] ? true : false;
        
        add_filter( 'plugins_api', array( $this, 'update_info' ), 10, 3 );
        
        add_filter( 'pre_set_site_transient_update_plugins', array( $this, 'push_update' ), 10, 1 );
        
        add_action( 'admin_notices', array( $this, 'admin_notice' ) );

    }
    
    /**
     * Performs REST API request to the activation server
     * 
     * @param string $path The endpoint in the activation server
     * @param array $body The data to be passed to the server
     * 
     * @return object The REST API response if there is no errors, bool otherwise
     */
    public function request( $path, $body = null ){
        
        $url = self::API_SERVER . $path;
        
        if( $this->dev ){
            $url = 'http://localhost/wordpress/wp-json/aw-api/v1/' . $path;
        }
        
        $raw_response = wp_remote_post( $url, array(
            'headers' => array('Content-Type' => 'application/json; charset=utf-8'),
            'timeout' => 10,
            'body' => json_encode( $body )
        ));
        
        // wp error
        if( is_wp_error( $raw_response ) ) {
            return false;
        }
        
        // http error
        if( wp_remote_retrieve_response_code( $raw_response ) != 200 ) {
            return false;
        }
        
        // decode response
        $json = json_decode( wp_remote_retrieve_body( $raw_response ) );
        
        // allow non json value
        if( $json === null ) {
            return wp_remote_retrieve_body( $raw_response );
        }
        
        return $json;
        
    }
    
    /**
     * Retrieves the saved license from the db
     * 
     * @return bool If no license saved, array otherwise
     */
    public function get_license(){
        
        $license = get_option( $this->plugin_prefix . 'license' );
        
        if( $license ){
            
            $license = maybe_unserialize( base64_decode( $license ) );
            
            return wp_parse_args( $license, array(
                'key' => false,
                'status' => false,
                'expiry_date' => false
            ));
            
        }else{
            return false;
        }
        
    }

    /**
     * Saves the license information received to the db
     * 
     * @return The update option status
     */
    public function update_license( $license = array() ){
        
        $license = base64_encode( maybe_serialize( $license ) );
        
        return update_option( $this->plugin_prefix . 'license', $license );
        
    }

    /**
     * Deletes the license key from the db
     * 
     * @return The delete option status
     */
    public function delete_license(){
        
        return delete_option( $this->plugin_prefix . 'license' );
        
    }

    /**
     * Checks whether the license key is set or not
     * 
     * @return bool The license key is set status
     */
    public function is_license_set(){
        
        $license = $this->get_license();
        return ( $license && $license[ 'key' ] ) ? true : false;
        
    }

    /**
     * Checks whether the license has expired or not if it is set
     * 
     * @return bool The license expiry status
     */
    public function has_license_expired(){

        if( $this->is_license_set() ){
            $license = $this->get_license();
            $license_expiry = $license[ 'expiry_date' ];

            if( time() > strtotime( $license_expiry ) ){
                return true;
            }else{
                return false;
            }

        }else{
            return false;
        }

    }

    /**
     * Displays the license information, activation and deactivation page, form
     */
    public function license_page(){

        $this->handle_license_form();
        $license_manager_version = $this->license_manager_version;

        $license = $this->get_license();
        $is_license_set = $this->is_license_set();
        $license_key = $is_license_set ? $license[ 'key' ] : '';
        $license_expiry = $is_license_set ? $license[ 'expiry_date' ] : '';
        
        $p_title = $this->plugin_name;
        $p_nonce = $this->plugin_prefix . 'license_form_nonce';
        $p_version = $this->plugin_version;
        
        $update_info = array();
        if( $is_license_set ){
            $transient_name = $this->plugin_prefix . 'update_info';
            $update_info = get_transient( $transient_name );
            $update_info = is_object( $update_info ) ? (array) $update_info : array();
        }
        
        include_once( 'views/license-page.php' );
        
    }
    
    /**
     * Handles the license activation/deactivation form submission
     */
    public function handle_license_form(){
        
        if( !$_POST || !check_admin_referer( $this->plugin_prefix . 'license_form_nonce' ) ){
            return false;
        }
        
        $given_license_key = sanitize_text_field( isset( $_POST[ 'license_key' ] ) ? $_POST[ 'license_key' ] : '' );
        
        if( $_POST[ 'license_action' ] == 'activate' ){
            $this->activate_license( $given_license_key );
        }
        
        if( $_POST[ 'license_action' ] == 'deactivate' ){
            $this->deactivate_license();
        }
        
        if( $_POST[ 'license_action' ] == 'refresh' ){
            $this->refresh_license();
        }

    }
    
    /**
     * Activates the license key for the domain
     * 
     * @param string $key The license key
     * 
     * @return bool The activation status
     */
    public function activate_license( $key ){
        
        $response = $this->request( 'license', array(
            'action' => 'activate',
            'license_key' => $key,
            'domain' => $this->plugin_domain,
            'product_name' => $this->plugin_id,
            'version' => $this->plugin_version
        ));
        
        if( !is_object( $response ) ){
            $this->show_notice( 'License API server is currently not available. Please try again later.', 'error' );
            return false;
        }
        
        if( $response->result == 'failed' ){
            $this->show_notice( $response->message, 'error' );
            return false;
        }
        
        if( $response->result == 'success' ){
            $data = (array) $response->data;
            
            if( $this->update_license( $data ) ){
                $this->show_notice( $response->message, 'success' );
                $this->delete_transients();
            }else{
                $this->show_notice( 'Failed to save license information to DB', 'error' );
            }
            return true;
        }
        
        $this->show_notice( 'Received unsupported message from the activation server', 'error' );
        return false;
        
    }
    
    /**
     * Deactivates the license for the domain after informing the activation server
     * 
     * @return bool The deactivation status
     */
    public function deactivate_license(){
        
        $license = $this->get_license();
        
        if( !$this->is_license_set() ){
            $this->show_notice( 'License is not active to deactivate', 'error' );
            return false;
        }
        
        $response = $this->request( 'license', array(
            'action' => 'deactivate',
            'license_key' => $license[ 'key' ],
            'domain' => $this->plugin_domain,
            'product_name' => $this->plugin_id,
            'version' => $this->plugin_version
        ));
        
        if( !is_object( $response ) ){
            $this->show_notice( 'License API server is currently not available. Please try again later.', 'error' );
            return false;
        }
        
        if( $response->result == 'failed' ){
            $this->show_notice( $response->message, 'error' );
            return false;
        }
        
        if( $response->result == 'success' ){
            $data = (array) $response->data;
            
            if( $this->delete_license() ){
                $this->show_notice( $response->message, 'success' );
                $this->delete_transients();
            }else{
                $this->show_notice( 'Failed to save license information to DB', 'error' );
            }
            return true;
        }
        
        $this->show_notice( 'Received unsupported message from the activation server', 'error' );
        return false;
        
    }
    
    /**
     * Refreshes the license
     * 
     * @return bool The refresh status
     */
    public function refresh_license(){

        $license = $this->get_license();
        
        if( !$this->is_license_set() ){
            $this->show_notice( 'Please apply a license to refresh', 'error' );
            return false;
        }

        $response = $this->request( 'license', array(
            'action' => 'get_info',
            'license_key' => $license[ 'key' ],
            'domain' => $this->plugin_domain,
            'product_name' => $this->plugin_id,
            'version' => $this->plugin_version
        ));

        if( !is_object( $response ) ){
            $this->show_notice( 'License API server is currently not available. Please try again later.', 'error' );
            return false;
        }
        
        if( $response->result == 'failed' ){
            $this->show_notice( $response->message, 'error' );
            return false;
        }
        
        if( $response->result == 'success' ){
            $data = (array) $response->data;
            
            $license[ 'expiry_date' ] = $data[ 'expiry_date' ];
            $license[ 'domains_allowed' ] = $data[ 'domains_allowed' ];

            if( $this->update_license( $license ) ){
                $this->show_notice( 'Successfully refreshed license.', 'success' );
                $this->delete_transients();
            }else{
                $this->show_notice( 'License has not changed.', 'error' );
            }

            return true;
        }
        
        $this->show_notice( 'Received unsupported message from the activation server', 'error' );
        return false;

    }

    /**
     * The plugin information page when there is an update available.
     * 
     * @return object The plugin information object populated for the given slug
     */
    public function update_info( $res, $action, $args ){
        
        if( $action !== 'plugin_information' )
            return false;
        
        if( $args->slug !== $this->plugin_slug )
            return $res;
        
        $data = $this->check_update();
        
        if( !$data ){
            return false;
        }
        
        $res = new \stdClass();
        $res->name = $data->name;
        $res->slug = $data->slug;
        $res->version = $data->version;
        $res->tested = $data->tested;
        $res->requires = $data->requires;
        $res->author = '<a href="' . $data->author_homepage . '">' . $data->author . '</a>';
        $res->author_profile = 'https://profiles.wordpress.org/vaakash';
        $res->download_link = $data->download_url;
        $res->trunk = $data->download_url;
        $res->last_updated = $data->last_updated;
        $res->sections = array(
            'changelog' => $data->sections->changelog
        );

        $res->banners = array(
            'high' => $data->banner
        );
        
        return $res;
        
    }
    
    /**
     * Adds the plugin latest update information to the WordPress updates list after getting the latest information from the API server.
     * 
     * @return object The transient array which has the list of plugin updates information
     */
    public function push_update( $transient ){
        
        if ( empty( $transient->checked ) ){
            return $transient;
        }
        
        $data = $this->check_update();
        
        if( !$data ){
            return $transient;
        }
        
        if( !( version_compare( $this->plugin_version, $data->version, '<' ) && version_compare( get_bloginfo('version'), $data->requires, '>=' ) ) ){
            return $transient;
        }
        
        $res = new \stdClass();
        $res->slug = $this->plugin_slug;
        $res->plugin = $this->plugin_file;
        $res->new_version = $data->version;
        $res->tested = $data->tested;
        $res->package = $data->download_url;
        $res->url = $data->author_homepage;
        $res->icons = array(
            '2x' => $data->icon
        );
        $res->banners = array(
            '2x' => $data->banner
        );
        
        $transient->response[ $res->plugin ] = $res;
        
        return $transient;
        
    }
    
    /**
     * Requests the API server for the latest update information when license key is set
     * 
     * @return object The plugin's latest update download information
     */
    public function check_update(){
        
        $license = $this->get_license();
        $transient_name = $this->plugin_prefix . 'update_info';
        $last_attempt_failed = $this->plugin_prefix . 'last_attempt_failed';
        
        if( !$this->is_license_set() )
            return false;

        if( $this->has_license_expired() ){
            $refresh_transient_name = $this->plugin_prefix . 'last_refresh';
            $was_refreshed = get_transient( $refresh_transient_name );
            if( $was_refreshed != 'yes' ){
                ob_start();
                $this->refresh_license();
                ob_end_clean();
                set_transient( $refresh_transient_name, 'yes', DAY_IN_SECONDS * 7 );
            }
        }

        if( isset( $_GET[ 'force-check' ] ) ){
            $this->delete_transients();
        }
        
        if( get_transient( $last_attempt_failed ) == 'yes' ){
            return false;
        }

        $data = get_transient( $transient_name );
        
        if( is_object( $data ) ){
            return $data;
        }
        
        // Request for update info since transient is not set
        $response = $this->request( 'updates', array(
            'license_key' => $license[ 'key' ],
            'domain' => $this->plugin_domain,
            'product_name' => $this->plugin_id,
            'product_version' => $this->plugin_version
        ));
        
        if( !is_object( $response ) || $response->result != 'success' ){
            set_transient( $last_attempt_failed, 'yes', DAY_IN_SECONDS * 7 );
            return false;
        }
        
        $data = $response->data;
        set_transient( $transient_name, $data, DAY_IN_SECONDS * 7 );
        
        return $data;
        
    }
    
    /**
     * Returns the domain name of the site without http
     * 
     * @return string The domain name of the site
     */
    public function domain_url(){
        
        $url = home_url();
        return str_replace( array( 'http://', 'https://' ), '', $url );
        
    }
    
    /**
     * Deletes the update information transient stored for the plugin
     */
    public function delete_transients(){
        delete_transient( $this->plugin_prefix . 'update_info' );
        delete_transient( $this->plugin_prefix . 'last_attempt_failed' );
    }
    
    /**
     * The HTML wrapper tag for the admin notice
     */
    public function show_notice( $msg, $type = 'info' ){
        echo '<div class="notice notice-' . $type . ' is-dismissible"><p>' . $msg . '</p></div>';
    }
    
    /**
     * The admin notice for applying license key and on expiry
     */
    public function admin_notice(){

        $msg = '';
        $type = '';
        $dismiss_for = 15 * DAY_IN_SECONDS;
        $transient_name = $this->plugin_prefix . 'dismiss_notice';
        $defaults = array(
            'expiry' => false,
            'apply' => false
        );

        $dismiss_notice = wp_parse_args( get_transient( $transient_name ), $defaults );

        if( isset( $_GET[ 'awplm_dismiss_notice' ] ) && $_GET[ 'awplm_dismiss_notice' ] == $this->plugin_id ){
            if( check_admin_referer( 'dismiss_notice' ) ){
                $dismiss_type = isset( $_GET[ 'awplm_dismiss_type' ] ) ? $_GET[ 'awplm_dismiss_type' ] : false;
                if( in_array( $dismiss_type, array( 'expiry', 'apply' ) ) ){
                    $dismiss_notice[ $dismiss_type ] = true;
                    set_transient( $transient_name, $dismiss_notice, $dismiss_for );
                }
            }
        }

        if( $this->is_license_set() ){

            $license = $this->get_license();
            $license_expiry = $license['expiry_date'];
            $expiry_relative = $this->relative_time( $license_expiry );
            $expiry_relative = $expiry_relative ? $expiry_relative : $license_expiry;
            $renew_url = add_query_arg( 'license_key', $license[ 'key' ], 'https://www.aakashweb.com/renew' );

            if( $this->has_license_expired() && !$dismiss_notice[ 'expiry' ] ){

                $dismiss_url = add_query_arg(array(
                    'awplm_dismiss_type' => 'expiry',
                    'awplm_dismiss_notice' => $this->plugin_id,
                    '_wpnonce' => wp_create_nonce( 'dismiss_notice' )
                ));

                $type = 'error';
                $msg = sprintf( '
                <p><strong>%s</strong> plugin license has expired <strong title="%s">%s</strong> ago. Please renew the license for continued plugin updates.</p>
                <p><a href="%s" class="button button-primary" target="_blank">Renew license</a>
                <a href="%s" class="button">Dismiss</a>
                </p>',
                $this->plugin_name, $license_expiry, $expiry_relative, $renew_url, $dismiss_url );

            }

        }else{

            if( !$dismiss_notice[ 'apply' ] && !( isset( $_GET[ 'page' ] ) && $_GET[ 'page' ] == $this->license_page ) ){

                $dismiss_url = add_query_arg(array(
                    'awplm_dismiss_type' => 'apply',
                    'awplm_dismiss_notice' => $this->plugin_id,
                    '_wpnonce' => wp_create_nonce( 'dismiss_notice' )
                ));

                $type = 'info';
                $msg = sprintf(
                    '<p><strong>%s</strong> plugin has been installed. Please apply the license received to get plugin updates</p>
                    <p><a href="%s" class="button button-primary">Apply license</a>
                    <a href="%s" class="button">Dismiss</a>
                    </p>',
                $this->plugin_name, admin_url( 'admin.php?page=' . $this->license_page ), $dismiss_url );

            }

        }

        if( !empty( $msg ) ){
            echo '<div class="notice notice-' . $type . ' is-dismissible">' . $msg . '</div>';
        }

        if( $this->dev ){
            echo '<p style="position: fixed; top: 7px; right: 150px; background: red; z-index: 99999; margin: 0">' . $this->plugin_id . ' - dev mode</p>';
        }

    }

    /**
     * Converts formatted time to relative time
     * 
     * @return string The relative time on successful parse or false otherwise.
     */
    public function relative_time( $date ){

        $d = \DateTime::createFromFormat( 'Y-m-d H:i:s', $date );
        if ($d === false) {
            return false;
        } else {
            $timestamp = $d->getTimestamp();
            return human_time_diff( $timestamp );
        }

    }

}

?>